/*!
* \file       LWDSApi.h
* \brief      LWDSApi.h
* \details    DS设备SDK：API头文件
*/


#ifndef LWDSAPI_H
#define LWDSAPI_H

#include "LWDSType.h"


/*!
 * \brief  直接获取指定设备的单帧数据
 * \param  deviceIp 指定的设备IP
 * \param  frame 存储单帧数据的自定义二维数组
 * \return 获取结果 \see LWDSReturnCode
 * \retval 没有找到指定IP对应的设备           LWDS_RETURN_DO_NOT_FIND_DEVICE
 * \retval 获取错误                         LWDS_RETURN_GET_DATA_ERROR
 * \retval 获取成功                         LWDS_RETURN_OK
 */
LW_DS_C_API LWDSReturnCode LWDSGetFrame(const char* deviceIp, LWDSFrameData frame);


/*!
 * \brief  注册回调函数，用于更新单帧的数据
 * \param  deviceIp 指定的设备IP
 * \param  FrameUpdateCallback 单帧数据更新函数的指针
 * \return 注册结果 \see LWDSReturnCode
 * \retval 没有找到指定IP对应的设备           LWDS_RETURN_DO_NOT_FIND_DEVICE
 * \retval 注册错误                         LWDS_RETURN_REGISTER_FUNCTION_ERROR
 * \retval 注册成功                         LWDS_RETURN_OK
 */
LW_DS_C_API LWDSReturnCode LWDSRegisterCallbackUpdateFrame(const char* deviceIp,void (*FrameUpdateCallback)(LWDSFrameData ));


/*!
 * \brief  注册回调函数，用于更新IMU的数据
 * \param  deviceIp 指定的设备IP
 * \param  ImuUpdateCallback IMU数据更新函数的指针
 * \return 注册结果 \see LWDSReturnCode
 * \retval 没有找到指定IP对应的设备           LWDS_RETURN_DO_NOT_FIND_DEVICE
 * \retval 注册错误                         LWDS_RETURN_REGISTER_FUNCTION_ERROR
 * \retval 注册成功                         LWDS_RETURN_OK
 */
LW_DS_C_API LWDSReturnCode LWDSRegisterCallbackUpdateImu(const char* deviceIp,void (*ImuUpdateCallback)(LWDSImuData));


/*!
 * \brief  监听本地端口,开始接收设备数据
 * \param  localPort 本地端口号
 * \return 连接结果 \see LWDSReturnCode
 */
LW_DS_C_API LWDSReturnCode LWDSStartReceiveData(uint32_t localPort);

/*!
 * \brief  停止监听本地端口
 * \return 停止结果 \see LWDSReturnCode
 */
LW_DS_C_API LWDSReturnCode LWDSStopReceiveData();


/*!
 * \brief  连接设备,开始接收指定设备的数据
 * \param  deviceIp    设备地址
 * \param  devicePort  设备端口号
 * \return 设备连接结果 \see LWDSReturnCode
 * \retval 连接设备成功                                        LWDS_RETURN_OK
 * \retval 连接设备成功,但是设备已创建,此次操作仅变更设备端口       LWDS_RETURN_DEVICE_EXIST
 * \retval 连接设备失败,IP信息或端口异常                         LWDS_RETURN_TARGET_IPPORT_NULL
 */
LW_DS_C_API LWDSReturnCode LWDSConnectDevice(const char* deviceIp, uint32_t devicePort);

/*!
 * \brief  断开设备的连接,停止接收这个设备的数据
 * \param  deviceIp    设备地址
 * \return 设备断开结果 \see LWDSReturnCode
 * \retval 断开设备成功               LWDS_RETURN_OK
 * \retval 没有找到指定IP对应的设备    LWDS_RETURN_DO_NOT_FIND_DEVICE
 */
LW_DS_C_API LWDSReturnCode LWDSDisconnectDevice(const char* deviceIp);


/*!
 * \brief  断开设备的连接,停止接收这个设备的数据
 * \attention 飞点滤波的阈值值越小,滤波效果越明显. 目前建议默认设置为45左右即可,根据距离可适当调整
 *
 * \param  deviceIp     设备地址
 * \param  enable       飞点滤波
 * \param  threshold    滤波阈值 (1-200)
 * \return 滤波设置结果 \see LWDSReturnCode
 * \retval 没有找到指定IP对应的设备      LWDS_RETURN_DO_NOT_FIND_DEVICE
 * \retval 阈值超过上限                 LWDS_RETURN_PARAM_EXCEEDS_LIMIT
 * \retval 设置飞点滤波成功              LWDS_RETURN_OK
 */
LW_DS_C_API LWDSReturnCode LWDSSetFlyingFilter(const char *deviceIp, bool enable, uint32_t threshold);


/*!
 * \brief 设置当前的网络信息
 * \param targetIp      设置目标IP
 * \param linkTimeout   设置超时时间
 * \param ipInfo        IP信息结构体 \see LWDSIpInfo
 * \return 设置结果 \see LWDSReturnCode
 * \retval 设置错误     LWDS_RETURN_SET_ADDR_ERROR
 * \retval 设置成功     LWDS_RETURN_OK
 */
LW_DS_C_API LWDSReturnCode LWDSSetNetInfo(const char *targetIp,int linkTimeout, LWDSIpInfo ipInfo);

/*!
 * \brief 读取当前的网络信息
 * \param targetIp      设置目标IP
 * \param linkTimeout   设置超时时间
 * \param ipInfo        IP信息结构体指针 \see LWDSIpInfo
 * \return 读取结果 \see LWDSReturnCode
 * \retval 读取错误     LWDS_RETURN_GET_ADDR_ERROR
 * \retval 读取成功     LWDS_RETURN_OK
 */
LW_DS_C_API LWDSReturnCode LWDSGetNetInfo(const char *targetIp,int linkTimeout, LWDSIpInfo *ipInfo);


/*!
 * \brief 获取当前返回码对应的描述文本
 * \details 二次开发时,可以拷贝这个函数,自定义其他风格的打印文本
 * \param code      返回码
 * \return 对应的描述文本
 */
inline const char * LWDSReturnCodeToString(enum LWDSReturnCode code) {
    std::string codeStr;

    //宏定义开关,用于控制打印时,是否附加返回码的数值
#define STRING_WITH_CODE_VALUE
#ifdef STRING_WITH_CODE_VALUE
    //打印时附加返回码的数值
    codeStr.append("[").append(std::to_string(code)).append("] ");
#endif
#undef STRING_WITH_CODE_VALUE

    switch (code) {
        //错误码区段
        case LWDS_RETURN_LOCAL_PORT_NULL:
            return codeStr.append("Local port is null").c_str();
        case LWDS_RETURN_TARGET_IP_PORT_ERROR:
            return codeStr.append("Invalid target address").c_str();
        case LWDS_RETURN_BIND_FAILD:
            return codeStr.append("Port binding failed").c_str();
        case LWDS_RETURN_SHAKE_HAND_FAILD:
            return codeStr.append("Handshake protocol failure").c_str();
        case LWDS_RETURN_OPEN_FILE_FAILD:
            return codeStr.append("File open failure").c_str();
        case LWDS_RETURN_READ_FILE_FAILD:
            return codeStr.append("File read failure").c_str();
        case LWDS_RETURN_CLOSE_SOCK_FAILD:
            return codeStr.append("Socket close failed").c_str();
        case LWDS_RETURN_INIT_SOCK_FAILD:
            return codeStr.append("Socket initialization failed").c_str();
        case LWDS_RETURN_GET_DATA_ERROR:
            return codeStr.append("Data acquisition failed").c_str();
        case LWDS_RETURN_REGISTER_FUNCTION_ERROR:
            return codeStr.append("Function registration failed").c_str();
        case LWDS_RETURN_SET_ADDR_ERROR:
            return codeStr.append("Network config setup failed").c_str();
        case LWDS_RETURN_GET_ADDR_ERROR:
            return codeStr.append("Network config read failed").c_str();
        case LWDS_RETURN_DO_NOT_FIND_DEVICE:
            return codeStr.append("Target device not found").c_str();
        case LWDS_RETURN_PARAM_OUT_OF_BOUND:
            return codeStr.append("Parameter out of bounds").c_str();
        case LWDS_RETURN_ERROR:
            return codeStr.append("Error").c_str();

        //状态码区段
        case LWDS_RETURN_OK:
            return codeStr.append("OK").c_str();
        case LWDS_RETURN_DEVICE_EXIST:
            return codeStr.append("Device already exists").c_str();
        default:
            return codeStr.append("Undefined code").c_str();
    }
}


#endif
