/*!
* \file       LWDSType.h
* \brief      LWDSType.h
* \details    DS设备SDK：类型信息头文件
*/


#ifndef LWDSTYPE_H
#define LWDSTYPE_H


#define LW_DS_C_EXPORT

#if defined(_WIN32)
#ifdef LW_DS_C_EXPORT
        #define LW_DS_API __declspec(dllexport)
#else
#define LW_DS_API __declspec(dllimport)
#endif
#else
#define LW_DS_API __attribute__((visibility("default")))
#endif

#ifdef __cplusplus
#define LW_DS_C_API //extern "C" LW_DS_API
#else
    #define LW_DS_C_API LW_DS_API
#endif

#include <cstdint>

//当前的SDK版本
#define LW_DS_SDK_VERSION "0.0.3.0"

//单字节对齐
#pragma pack(push,1)

//像素宽度
#define LW_DS_WIDTH 360
#define LW_DS_HEIGHT 150

/*!
 * \brief 设备的返回码
 * \attention 返回码有正负区分
 * \li 返回码为负数,通常代表函数执行失败
 * \li 返回码等于0,代表成功
 * \li 返回码大于0,代表额外的成功状态,用于对函数内的操作进行提示
 */
enum LWDSReturnCode {
    LWDS_RETURN_LOCAL_PORT_NULL         = -2000, // 本地端口为空
    LWDS_RETURN_TARGET_IP_PORT_ERROR    = -1999, // 目标IP错误或端口为空
    LWDS_RETURN_BIND_FAILD              = -1998, // 端口绑定失败
    LWDS_RETURN_SHAKE_HAND_FAILD        = -1997, // 握手失败
    LWDS_RETURN_OPEN_FILE_FAILD         = -1996, // 打开文件失败
    LWDS_RETURN_READ_FILE_FAILD         = -1995, // 读取文件失败
    LWDS_RETURN_CLOSE_SOCK_FAILD        = -1994, // 关闭套接字失败
    LWDS_RETURN_INIT_SOCK_FAILD         = -1993, // 初始化套接字失败
    LWDS_RETURN_GET_DATA_ERROR          = -1992, // 获取数据失败
    LWDS_RETURN_REGISTER_FUNCTION_ERROR = -1991, // 注册函数失败
    LWDS_RETURN_SET_ADDR_ERROR          = -1990, // 设置网络信息失败
    LWDS_RETURN_GET_ADDR_ERROR          = -1989, // 读取网络信息失败
    LWDS_RETURN_DO_NOT_FIND_DEVICE      = -1988, // 设备对象不存在
    LWDS_RETURN_PARAM_OUT_OF_BOUND      = -1987, // 参数超出限制
    LWDS_RETURN_ERROR                   = -1,    // 通用错误标识
    LWDS_RETURN_OK                      = 0,     // 正确
    LWDS_RETURN_DEVICE_EXIST            = 1,     // 设备对象已存在,无需重复创建
};



/*!
 * \brief 一次计算的运行结果
 * 包含状态码和文本描述,便于提供更精确的调试信息
 */
struct LWDSAlgorithmInfo {
    int code;       //完成计算后的返回码
    char* message;  //计算中的日志信息
};

/*!
 * \brief 算法模块的状态码
 */
enum LWDSAlgorithmCode {
    LWDS_ALGO_NOT_INIT = -100, //算法模块未初始化
    LWDS_ALGO_ERROR,           //算法模块异常
    LWDS_ALGO_FATAL,           //算法模块故障
    LWDS_ALGO_OK = 0,          //计算正确
};


/*!
 * \brief 网络信息结构体
 */
struct LWDSIpInfo {
    uint32_t lidarIp;
    uint32_t hostIp;
    uint16_t lidarPort;
    uint16_t hostPort;
    uint16_t reserved[16];
};

/*!
 * \brief 点云信息的元数据
 */
struct LWDSVector3f {
    float x;
    float y;
    float z;
};


/*!
 * \brief 点数据结构体
 */
struct LWDSPointData {
    uint16_t     row;          // 行号
    uint16_t     col;          // 列号
    float        distance;     // 距离单位: cm
    uint32_t     reflectivity; // 反射率
    LWDSVector3f point3d;      // 3D坐标值
    uint64_t     timeStamp;    // 时间单位:us
    uint64_t     timeType;     // 时间类型
};


/*!
 * \brief 单帧数据以此方式进行传递
 */
typedef LWDSPointData LWDSFrameData[LW_DS_HEIGHT][LW_DS_WIDTH];


/*!
 * \brief IMU数据结构体
 */
struct LWDSImuData {
    uint8_t     crc8_1;             ///< CRC
    uint8_t     crc8_2;             ///< CRC
    uint8_t     version;            ///< SDK版本
    uint16_t    length;             ///< 数据长度
    uint8_t     timeType;           ///< 时间类型
    uint64_t    timeStamp;          ///< 单位: ns
    uint8_t     reserved[2];        ///< 保留位
    float       xAxisACC;           ///< X轴加速度
    float       yAxisACC;           ///< y轴加速度
    float       zAxisACC;           ///< z轴加速度
    float       xAxisGyro;          ///< 陀螺仪X轴角速度
    float       yAxisGyro;          ///< 陀螺仪y轴角速度
    float       zAxisGyro;          ///< 陀螺仪z轴角速度
};


#pragma pack(pop)


#endif //LWDSTYPE_H
