﻿#include <iostream>
#include <thread>
#include <chrono>
#include <iostream>
#if (_WINDLL||_WINDOWS||_WIN32)
#include <winsock2.h>
#else
#include <arpa/inet.h>
#endif

#include <cstring>
#include <iomanip>

#include "LWDSApi.h"

constexpr double kFpsLimit = 8.5;

/*!
 * \brief 帧数据更新时的回调函数
 * \param frame
 */
void frameDataCallback(LWDSFrameData frame) {
    //记录起始的时间信息
    static  std::chrono::time_point<std::chrono::high_resolution_clock> lastTime =
        std::chrono::high_resolution_clock::now();

    //当前时间
    auto currentTime = std::chrono::high_resolution_clock::now();
    //时间差
    double deltaTime = std::chrono::duration<double>(currentTime - lastTime).count();

    //帧率计数上限
    constexpr int kFrameRateCheck = 10;

    //帧率计数
    static int frameCount = 0;
    frameCount++;

    //到达上限时 开始判断
    if (frameCount > kFrameRateCheck) {
        auto fps = frameCount / deltaTime;

        auto now = std::chrono::system_clock::now();
        std::time_t now_time = std::chrono::system_clock::to_time_t(now);
        std::tm local_time = *std::localtime(&now_time);
        if (fps < kFpsLimit) {
            std::cerr << std::put_time(&local_time, "%A, %B %d, %Y %H:%M:%S") << " Dev 1 Low Frame Rate : " << fps << std::endl;
        } else {
            std::cout << std::put_time(&local_time, "%A, %B %d, %Y %H:%M:%S") << " Dev 1 Frame Rate : " << fps << std::endl;
        }
        //更新时间/计数
        lastTime = currentTime;
        frameCount = 0;
    }
}

void frameDataCallback_Dev2(LWDSFrameData frame) {
    //记录起始的时间信息
    static  std::chrono::time_point<std::chrono::high_resolution_clock> lastTime =
        std::chrono::high_resolution_clock::now();

    //当前时间
    auto currentTime = std::chrono::high_resolution_clock::now();
    //时间差
    double deltaTime = std::chrono::duration<double>(currentTime - lastTime).count();

    //帧率计数上限
    constexpr int kFrameRateCheck = 10;

    //帧率计数
    static int frameCount = 0;
    frameCount++;

    //到达上限时 开始判断
    if (frameCount > kFrameRateCheck) {
        auto fps = frameCount / deltaTime;

        auto now = std::chrono::system_clock::now();
        std::time_t now_time = std::chrono::system_clock::to_time_t(now);
        std::tm local_time = *std::localtime(&now_time);
        if (fps < kFpsLimit) {
            std::cerr << std::put_time(&local_time, "%A, %B %d, %Y %H:%M:%S") << " Dev 2 Low Frame Rate : " << fps << std::endl;
        } else {
            std::cout << std::put_time(&local_time, "%A, %B %d, %Y %H:%M:%S") << " Dev 2 Frame Rate : " << fps << std::endl;
        }
        //更新时间/计数
        lastTime = currentTime;
        frameCount = 0;
    }
}



/*!
 * \brief IMU数据更新时的回调函数
 * \param imu
 */
void imuDataCallback(LWDSImuData imu) {
    // std::cout << "Get new imu data!" << std::endl;
}

//默认的设备地址
#define DEFAULT_DEV_IP "192.168.137.200"

//第二个设备
#define SECOND_DEV_IP "192.168.137.201"

//默认的主机地址
#define DEFAULT_HOST_IP "192.168.137.100"


//可以通过修改宏定义的实现,来快速切换实时数据采集/回调函数采集的模式

//实时采集数据
// #define REAL_TIME_GET_DATA
#ifndef REAL_TIME_GET_DATA
//通过回调采集数据
#define CALL_BACK_GET_DATA
#endif


int main() {
    std::cout << "Start LuminWave DS Demo Process" << "\n" << std::endl;
    LWDSReturnCode ret;    //接收函数的状态码
    std::string    result; //返回状态的文本显示

    /// \brief 读取设备的网络信息 (需要有相应的下位机软件支持)
    LWDSIpInfo info;
    auto getNetInfo = LWDSGetNetInfo(DEFAULT_DEV_IP, 2000, &info);
    if (getNetInfo == LWDS_RETURN_OK) {
        in_addr ip_addr;
        ip_addr.s_addr = info.lidarIp;

        in_addr ip_addr2;
        ip_addr2.s_addr = info.hostIp;

        std::string lidar_ip_str;
        std::string host_ip_str;

        //打印当前获取的IP信息和端口号
        lidar_ip_str.assign(inet_ntoa(ip_addr));
        host_ip_str.assign(inet_ntoa(ip_addr2));
        std::cout << "Lidar ip:" << lidar_ip_str << std::endl;
        std::cout << "Host ip:" << host_ip_str << std::endl;
        std::cout << "Lidar port:" << info.lidarPort << std::endl;
        std::cout << "Host port:" << info.hostPort << std::endl;
    }

    /// \brief 修改当前指定的IP信息
    /// \waring 不建议修改端口信息
    /// \attention 多机模式下,建议在修改时,确保从机的网络信息中,主机的IP/端口信息一致
// #define TEST_IP_CHANGE_FUNCTION
#ifdef TEST_IP_CHANGE_FUNCTION
    //根据需要，修改IP，修改后建议设备重新上电
    info.hostIp = inet_addr("192.168.137.100");
    info.lidarIp = inet_addr("192.168.137.200");
    ret           = LWDSSetNetInfo(DEFAULT_DEV_IP, 200, info);
    result = LWDSReturnCodeToString(ret);//打印结果
    std::cout << "Change device ip info: " << result << "\n" << std::endl;
#endif


    /// \brief 设置本地端口,开始接收设备上传的数据
    /// \attention 在作业时,LWDSStartReceiveData 通常仅需调用一次,象征着当前已经开始监听
    ret           = LWDSStartReceiveData(6002);
    std::cout << "Start receive data: " << LWDSReturnCodeToString(ret) << "\n" << std::endl;

    /// \brief 连接默认设备, 使用对应IP和对应端口
    ret           = LWDSConnectDevice(DEFAULT_DEV_IP, 6001);
    std::cout << "Connect default device: " << LWDSReturnCodeToString(ret) << "\n" << std::endl;

    /// \brief 连接第二设备, 使用对应IP和对应端口
    /// \attention 连接设备时, 在确认当前已经有IP对应的情况下,会重设端口信息,并返回LWDS_RETURN_DEVICE_EXIST
    ret           = LWDSConnectDevice(SECOND_DEV_IP, 6001);
    std::cout << "Connect device 2: " << LWDSReturnCodeToString(ret) << "\n" << std::endl;


    /// \brief 若要使用回调函数获取数据，请先注册回调函数
    /// 注册回调函数成功后,当数据获取完毕时,将触发回调函数
    /// 这一步操作也会检查是否存在IP对应的设备
    /// \enum LWDS_RETURN_REGISTER_FUNCTION_ERROR    回调函数注册失败的返回码
    /// \enum LWDS_RETURN_OK                         回调函数注册成功时的返回码
#ifdef CALL_BACK_GET_DATA
    /// \brief 注册默认设备回调函数 用以计算帧率
    ret = LWDSRegisterCallbackUpdateFrame(DEFAULT_DEV_IP,frameDataCallback);
    std::cout << "Register default device update frame callback function: " << LWDSReturnCodeToString(ret) << "\n" << std::endl;

    /// \brief 注册第二设备回调函数 用以计算帧率
    ret = LWDSRegisterCallbackUpdateFrame(SECOND_DEV_IP,frameDataCallback_Dev2);
    std::cout << "Register device 2 update frame callback function: " << LWDSReturnCodeToString(ret) << "\n" << std::endl;

    /// \brief 注册回调函数 用以获取当前的IMU数据
    ret = LWDSRegisterCallbackUpdateImu(DEFAULT_DEV_IP,imuDataCallback);

    std::cout << "Register update imu callback function: " << LWDSReturnCodeToString(ret) << "\n" << std::endl;
#endif

    /// \brief 设置飞点滤波的使能 默认建议设置45
    ret = LWDSSetFlyingFilter(DEFAULT_DEV_IP,true,45);
    std::cout << "Set flying filter: " << LWDSReturnCodeToString(ret) << "\n" << std::endl;

    /// \brief 以下为测试代码段,持续等待,观察控制台中回调函数的输出结果
    /// \code
    //等待周期计数
    int waitingPeriod = 0;

    //最大等待周期
    constexpr int kMaxWaitingPeriod = 10000;
    std::cout << "Main thread start wait" << "!\n" << std::endl;

    /// \brief 主线程进入等待状态,此时若注册回调函数成功,那么可以看到持续的打印信息
    while (waitingPeriod < kMaxWaitingPeriod) {
        std::string getFrameResult;
        //测试实时采集数据的模式
#ifdef REAL_TIME_GET_DATA
        //获取帧数据
        auto frameData = new LWDSFrameData;

        ret = LWDSGetFrame(DEFAULT_DEV_IP,frameData);
        std::cout << "Get frame dev1: " << LWDSReturnCodeToString(ret) << "!\n" << std::endl;

        ret = LWDSGetFrame(SECOND_DEV_IP,frameData);
        std::cout << "Get frame dev2: " << LWDSReturnCodeToString(ret) << "!\n" << std::endl;
        delete frameData;
#endif
        std::this_thread::sleep_for(std::chrono::milliseconds(100));
        waitingPeriod++;
    }
    /// \code

    std::cout << "Ready to disconnect" << "!\n" << std::endl;

    //断开默认设备连接
    LWDSDisconnectDevice(DEFAULT_DEV_IP);

    //断开第二设备设备连接
    LWDSDisconnectDevice(SECOND_DEV_IP);

    //停止接收数据,一般在程序最后退出时调用
    LWDSStopReceiveData();
    return 0;
}

